<?php
session_start();
ob_start(); // Start output buffering
include 'db.php';
include 'header.php';

// Check if the user is logged in and is an Admin
if (!isset($_SESSION['user_id']) || $_SESSION['role_id'] != 1) {
    header('Location: login.php');
    exit;
}

// Pagination settings
$items_per_page = 10;
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $items_per_page;

// Search functionality
$searchTerm = isset($_GET['search']) ? '%' . $_GET['search'] . '%' : '';

// Fetch all delivered orders
$sqlDelivered = "SELECT o.id AS order_id, o.user_id, o.product_id, o.quantity, o.status, o.item_type,
                        p.name AS product_name, p.price AS product_price, p.stock,
                        d.discount_percentage, d.discount_type, d.discount_value,
                        (CASE 
                            WHEN d.discount_type = 'percentage' THEN (p.price * o.quantity * (d.discount_percentage / 100))
                            WHEN d.discount_type = 'fixed_amount' THEN (d.discount_value * o.quantity)
                            ELSE 0
                        END) AS discount_amount,
                        (p.price * o.quantity) - 
                        (CASE 
                            WHEN d.discount_type = 'percentage' THEN (p.price * o.quantity * (d.discount_percentage / 100))
                            WHEN d.discount_type = 'fixed_amount' THEN (d.discount_value * o.quantity)
                            ELSE 0
                        END) AS total_amount,
                        c.name AS company_name,
                        o.created_at, o.delivered_on_time
                 FROM orders o
                 JOIN products p ON o.product_id = p.id
                 LEFT JOIN discounts d ON p.id = d.product_id 
                 AND d.company_id = (SELECT company_id FROM users WHERE id = o.user_id)
                 LEFT JOIN users u ON o.user_id = u.id
                 LEFT JOIN companies c ON u.company_id = c.id
                 WHERE o.status = 'Delivered' AND (p.name LIKE :search OR c.name LIKE :search)
                 ORDER BY o.delivered_on_time DESC
                 LIMIT :limit OFFSET :offset";

$stmtDelivered = $pdo->prepare($sqlDelivered);
$stmtDelivered->bindValue(':search', $searchTerm, PDO::PARAM_STR);
$stmtDelivered->bindValue(':limit', $items_per_page, PDO::PARAM_INT);
$stmtDelivered->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmtDelivered->execute();
$deliveredOrders = $stmtDelivered->fetchAll(PDO::FETCH_ASSOC);

// Total delivered orders for pagination
$totalDeliveredOrders = $pdo->prepare("SELECT COUNT(*) FROM orders o
                                       JOIN products p ON o.product_id = p.id
                                       LEFT JOIN companies c ON (SELECT company_id FROM users WHERE id = o.user_id) = c.id
                                       WHERE o.status = 'Delivered' AND (p.name LIKE :search OR c.name LIKE :search)");
$totalDeliveredOrders->bindValue(':search', $searchTerm, PDO::PARAM_STR);
$totalDeliveredOrders->execute();
$totalDeliveredCount = $totalDeliveredOrders->fetchColumn();
$totalDeliveredPages = ceil($totalDeliveredCount / $items_per_page);

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Delivered Orders</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
</head>
<body>
<div class="container-fluid">
    <div class="row">
        <div class="col-md-12">
            <h1 class="mt-4">Delivered Orders</h1>

            <form method="GET" class="mb-3">
                <input type="text" name="search" placeholder="Search by product or company" value="<?= htmlspecialchars($_GET['search'] ?? '') ?>" class="form-control" style="width: 300px; display: inline-block;">
                <button type="submit" class="btn btn-primary">Search</button>
            </form>

            <form method="POST">
                <h3>Delivered Orders</h3>
                <table class="table">
                    <thead>
                        <tr>
                            <th>Product Name</th>
                            <th>Company Name</th>
                            <th>Quantity</th>
                            <th>Order Type</th> <!-- Added Order Type column -->
                            <th>Status</th>
                            <th>Total Amount</th>
                            <th>Delivered On</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($deliveredOrders)): ?>
                            <tr>
                                <td colspan="7">No delivered orders found.</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($deliveredOrders as $order): ?>
                                <tr>
                                    <td><?= htmlspecialchars($order['product_name'] ?? '') ?></td>
                                    <td><?= htmlspecialchars($order['company_name'] ?? '') ?></td>
                                    <td><?= htmlspecialchars($order['quantity'] ?? '') ?></td>
                                    <td><?= htmlspecialchars($order['item_type'] ?? '') ?></td> <!-- Show Single or Box -->
                                    <td><?= htmlspecialchars($order['status'] ?? '') ?></td>
                                    <td><?= htmlspecialchars(number_format($order['total_amount'] ?? 0, 2)) ?> Rs.</td>
                                    <td><?= htmlspecialchars($order['delivered_on_time'] ?? '') ?></td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>

                <!-- Pagination for Delivered Orders -->
                <nav>
                    <ul class="pagination">
                        <?php for ($i = 1; $i <= $totalDeliveredPages; $i++): ?>
                            <li class="page-item <?= ($i == $page) ? 'active' : '' ?>">
                                <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($_GET['search'] ?? '') ?>"><?= $i ?></a>
                            </li>
                        <?php endfor; ?>
                    </ul>
                </nav>

            </form>
        </div>
    </div>
</div>
</body>
</html>
