<?php
session_start();
include 'db.php';
require_once('vendor/autoload.php'); // Include TCPDF for PDF generation

// Check if the user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];

// Fetch user details to get the company ID and role ID
$sql = "SELECT * FROM users WHERE id = ?";
$stmt = $pdo->prepare($sql);
$stmt->execute([$user_id]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);

$role_id = $user['role_id'];
$company_id = $user['company_id'];
$current_date = date('Y-m-d');

// Fetch all pending orders for the same company on the current date
$sqlOrders = "SELECT o.id AS order_id, o.quantity, o.total_price, p.name AS product_name, p.price
              FROM orders o
              JOIN products p ON o.product_id = p.id
              WHERE o.company_id = :company_id AND DATE(o.order_date) = :order_date AND o.status = 'pending'";

$stmtOrders = $pdo->prepare($sqlOrders);
$stmtOrders->execute([
    'company_id' => $company_id,
    'order_date' => $current_date
]);

$orders = $stmtOrders->fetchAll(PDO::FETCH_ASSOC);

// Handle the "Mark as Delivered" action
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['mark_as_delivered'])) {
    // Mark all pending orders for the company as delivered
    $sqlUpdate = "UPDATE orders SET status = 'delivered' WHERE company_id = :company_id AND DATE(order_date) = :order_date";
    $stmtUpdate = $pdo->prepare($sqlUpdate);
    $stmtUpdate->execute([
        'company_id' => $company_id,
        'order_date' => $current_date
    ]);

    // Generate PDF invoice
    $pdf = new TCPDF();
    $pdf->AddPage();

    // Set your company logo, name, and current date/time
    $pdf->Image('path_to_logo.png', 10, 10, 30);  // Adjust logo path, position, and size
    $pdf->SetFont('helvetica', 'B', 16);
    $pdf->Cell(0, 10, 'Your Company Name', 0, 1, 'C');
    $pdf->SetFont('helvetica', '', 12);
    $pdf->Cell(0, 10, 'Date: ' . date('Y-m-d H:i:s'), 0, 1, 'R');

    // Add a line break
    $pdf->Ln(10);

    // Add order details
    $total_amount = 0;
    $pdf->SetFont('helvetica', '', 10);
    foreach ($orders as $order) {
        $pdf->Cell(50, 10, 'Product: ' . $order['product_name'], 0, 0);
        $pdf->Cell(30, 10, 'Quantity: ' . $order['quantity'], 0, 0);
        $pdf->Cell(30, 10, 'Price: Rs. ' . number_format($order['price'], 2), 0, 0);
        $pdf->Cell(30, 10, 'Total: Rs. ' . number_format($order['total_price'], 2), 0, 1);
        $total_amount += $order['total_price'];
    }

    // Display the total amount at the bottom
    $pdf->Cell(0, 10, 'Total Amount: Rs. ' . number_format($total_amount, 2), 0, 1, 'R');

    // Output the PDF to the browser
    $pdf->Output('invoice.pdf', 'I');  // Display the invoice in the browser
    exit; // Ensure no further code is executed
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Dashboard</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
</head>
<body>
<div class="container">
    <h1 class="mt-4">Dashboard</h1>

    <?php if ($role_id == 1): ?>
        <h2>Administrator View</h2>
        <p>Manage users, companies, and view system-wide data.</p>
        <!-- Admin management links here -->
    <?php elseif ($role_id == 2): ?>
        <h2>Admin View</h2>
        <p>Manage products, view reports, and handle company-specific tasks.</p>
        <!-- Admin management links here -->
    <?php elseif ($role_id == 3): ?>
        <h2>Client View</h2>
        <p>View and manage personal information.</p>
    <?php endif; ?>

    <?php if (!empty($orders)): ?>
        <h3>Pending Orders</h3>
        <table class="table">
            <thead>
                <tr>
                    <th>Order ID</th>
                    <th>Product Name</th>
                    <th>Quantity</th>
                    <th>Price (Rs.)</th>
                    <th>Total (Rs.)</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($orders as $order): ?>
                    <tr>
                        <td><?= htmlspecialchars($order['order_id']) ?></td>
                        <td><?= htmlspecialchars($order['product_name']) ?></td>
                        <td><?= htmlspecialchars($order['quantity']) ?></td>
                        <td><?= htmlspecialchars(number_format($order['price'], 2)) ?></td>
                        <td><?= htmlspecialchars(number_format($order['total_price'], 2)) ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <form method="post">
            <button type="submit" name="mark_as_delivered" class="btn btn-success">Mark as Delivered</button>
        </form>
    <?php else: ?>
        <p>No pending orders for today.</p>
    <?php endif; ?>
</div>
</body>
</html>
