<?php
session_start();
include 'db.php';
include 'header.php';

// Check if the user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// Retrieve selected order IDs
$orderIds = isset($_POST['order_ids']) ? $_POST['order_ids'] : [];

if (empty($orderIds)) {
    echo "No orders selected for invoice generation.";
    exit;
}

// Fetch order details for selected orders, including discounts
$placeholders = implode(',', array_fill(0, count($orderIds), '?'));
$sql = "SELECT o.*, u.company_id, c.name AS company_name, p.name AS product_name, 
               p.price, c.logo AS company_logo, d.discount_value, d.discount_type
        FROM orders o 
        JOIN users u ON o.user_id = u.id 
        JOIN companies c ON u.company_id = c.id
        JOIN products p ON o.product_id = p.id 
        LEFT JOIN discounts d ON o.product_id = d.product_id AND u.company_id = d.company_id
        WHERE o.id IN ($placeholders)";

$stmt = $pdo->prepare($sql);
$stmt->execute($orderIds);
$orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Initialize totals
$grandTotal = 0;
$totalDiscountAmount = 0;

// Invoice generation logic
$invoiceNumber = uniqid("INV-");
$currentDate = date('Y-m-d H:i:s');

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Invoice</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <style>
        @media print {
            .btn { display: none; }
            body { margin: 0; padding: 0; font-size: 12pt; }
            .container { width: 100%; margin: 0; padding: 0; }
            .print-header { display: block; text-align: center; margin-bottom: 20px; }
            .print-header img { max-width: 150px; height: auto; }
            .table { border-collapse: collapse; width: 100%; }
            .table th, .table td { border: 1px solid #ddd; padding: 8px; text-align: left; }
            .table th { background-color: #f2f2f2; }
        }
    </style>
</head>
<body>
    <div class="container mt-4">
        <div class="print-header">
            <h2><?php echo htmlspecialchars($orders[0]['company_name'] ?? 'Your Company Name', ENT_QUOTES, 'UTF-8'); ?></h2>
            <img src="<?php echo htmlspecialchars($orders[0]['company_logo'] ?? 'path/to/default/logo.png', ENT_QUOTES, 'UTF-8'); ?>" alt="Company Logo" style="max-width: 150px; max-height: 100px;">
            <div class="text-right">
                <h4>Invoice No: <?= htmlspecialchars($invoiceNumber, ENT_QUOTES, 'UTF-8'); ?></h4>
                <h5><?= htmlspecialchars($currentDate, ENT_QUOTES, 'UTF-8'); ?></h5>
            </div>
        </div>

        <h1>Invoice</h1>

        <table class="table">
            <thead>
                <tr>
                    <th>Order ID</th>
                    <th>Product Name</th>
                    <th>Quantity</th>
                    <th>Total Amount (Rs)</th>
                    <th>Discount Amount (Rs)</th>
                    <th>Discount Rate Charge</th>
                    <th>Date</th>
                    <th>Company ID</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($orders as $order): ?>
                    <tr>
                        <td><?= htmlspecialchars($order['id'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
                        <td><?= htmlspecialchars($order['product_name'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
                        <td><?= htmlspecialchars($order['quantity'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
                        <td>
                            <?php
                            // Total amount calculation
                            $totalAmount = ($order['quantity'] ?? 0) * ($order['price'] ?? 0);
                            $grandTotal += $totalAmount;
                            // echo htmlspecialchars($totalAmount, ENT_QUOTES, 'UTF-8');
                            echo htmlspecialchars(number_format($totalAmount, 2), ENT_QUOTES, 'UTF-8');

                            ?>
                        </td>
                        <td>
                            <?php
                            // Calculate the discount amount
                            $discountValue = $order['discount_value'] ?? 0;
                            $quantity = $order['quantity'] ?? 0;

                            // Determine discount amount
                            if ($discountValue > 0) {
                                $discountAmount = $order['discount_type'] == 'fixed_amount' ? 
                                    $discountValue * $quantity : 
                                    ($totalAmount * $discountValue) / 100;
                            } else {
                                $discountAmount = $totalAmount; // No discount, so full amount is considered as discount
                            }

                            $totalDiscountAmount += $discountAmount; // Accumulate total discount amount

                            // echo htmlspecialchars($discountAmount, ENT_QUOTES, 'UTF-8');
                            echo htmlspecialchars(number_format($discountAmount, 2), ENT_QUOTES, 'UTF-8');

                            ?>
                        </td>
                        <td><?= htmlspecialchars($order['discount_value'] ?? '', ENT_QUOTES, 'UTF-8') ?> Rs</td>
                        <td><?= htmlspecialchars($order['created_at'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
                        <td><?= htmlspecialchars($order['company_id'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
            <tfoot>
                <tr>
                    <th colspan="4">Total Payable Amount</th>
                    <th><?= 
                    // htmlspecialchars($totalDiscountAmount, ENT_QUOTES, 'UTF-8');
                     htmlspecialchars(number_format($totalDiscountAmount, 2), ENT_QUOTES, 'UTF-8');

                     ?> Rs</th>
                    <th colspan="2"></th>
                </tr>
            </tfoot>
        </table>
        
        <div class="text-right mb-4">
            <button onclick="window.print();" class="btn btn-primary">Print</button>
        </div>

        <form method="POST" action="update_payment_status.php">
            <input type="hidden" name="invoice_number" value="<?= htmlspecialchars($invoiceNumber, ENT_QUOTES, 'UTF-8'); ?>">
            <!-- <button type="submit" name="status" value="paid" class="btn btn-success">Mark as Paid</button>
            <button type="submit" name="status" value="not_paid" class="btn btn-danger">Mark as Not Paid</button> -->
        </form>
    </div>
</body>
</html>
