<?php
session_start();
include 'db.php';
include 'header.php'; // Include the header with the menu

// Check if the user is logged in and is an Administrator
if (!isset($_SESSION['user_id']) || $_SESSION['role_id'] != 1) {
    header('Location: login.php');
    exit;
}

// Handle form submission to add/update discounts
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['apply_discount']) || isset($_POST['update_discount'])) {
        $company_id = $_POST['company_id'];
        $product_ids = $_POST['product_ids']; // Array of product IDs
        $discount_type = $_POST['discount_type']; // New field for discount type
        $discount_value = $_POST['discount_value']; // Can be percentage or amount
        $quantity = $_POST['quantity'];

        foreach ($product_ids as $product_id) {
            // Check if a discount already exists for the company and product
            $sqlCheck = "SELECT id FROM discounts WHERE company_id = ? AND product_id = ?";
            $stmtCheck = $pdo->prepare($sqlCheck);
            $stmtCheck->execute([$company_id, $product_id]);
            $existing_discount_id = $stmtCheck->fetchColumn();

            if ($existing_discount_id) {
                // If it exists, update the discount
                $sqlUpdate = "UPDATE discounts SET discount_type = ?, discount_value = ?, quantity = ? 
                              WHERE id = ?";
                $stmtUpdate = $pdo->prepare($sqlUpdate);
                $stmtUpdate->execute([$discount_type, $discount_value, $quantity, $existing_discount_id]);
            } else {
                // If it does not exist, insert a new discount
                $sqlInsert = "INSERT INTO discounts (company_id, product_id, discount_type, discount_value, quantity) 
                              VALUES (?, ?, ?, ?, ?)";
                $stmtInsert = $pdo->prepare($sqlInsert);
                $stmtInsert->execute([$company_id, $product_id, $discount_type, $discount_value, $quantity]);
            }
        }
        header('Location: manage_companies.php');
        exit;
    } elseif (isset($_POST['update_discount'])) {
        $discount_id = $_POST['discount_id'];
        foreach ($product_ids as $product_id) {
            // Update discount based on ID
            $sql = "UPDATE discounts SET company_id = ?, product_id = ?, discount_type = ?, discount_value = ?, quantity = ? 
                    WHERE id = ?";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$company_id, $product_id, $discount_type, $discount_value, $quantity, $discount_id]);
        }
        header('Location: manage_companies.php');
        exit;
    }
}

// Handle delete discount
if (isset($_GET['delete'])) {
    $discount_id = $_GET['delete'];

    // Delete discount
    $sql = "DELETE FROM discounts WHERE id = ?";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$discount_id]);

    header('Location: manage_companies.php');
    exit;
}

// Pagination setup
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$perPage = 10; // Number of items per page
$offset = ($page - 1) * $perPage;

// Fetch companies for discount management, excluding inactive ones
$sql = "SELECT * FROM companies WHERE is_active = 1";
$companies = $pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC);

// Fetch products with images
$sql = "SELECT * FROM products";
$products = $pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC);

// Fetch existing discounts with pagination
$sql = "SELECT d.*, c.name AS company_name, p.name AS product_name 
        FROM discounts d
        JOIN companies c ON d.company_id = c.id
        JOIN products p ON d.product_id = p.id
        LIMIT :offset, :perPage";
$stmt = $pdo->prepare($sql);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->bindValue(':perPage', $perPage, PDO::PARAM_INT);
$stmt->execute();
$existing_discounts = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch total count of discounts for pagination
$sqlCount = "SELECT COUNT(*) FROM discounts";
$total = $pdo->query($sqlCount)->fetchColumn();
$totalPages = ceil($total / $perPage);

// Check if editing a discount
$discount_to_edit = null;
if (isset($_GET['edit'])) {
    $discount_id = $_GET['edit'];

    // Fetch existing discount details
    $sql = "SELECT * FROM discounts WHERE id = ?";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$discount_id]);
    $discount_to_edit = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($discount_to_edit) {
        $product_ids = [$discount_to_edit['product_id']];
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Manage Discounts</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <link rel="stylesheet" href="./Style/css/Clogin.css">
<style>
    
body {
    background-color: #0e5091 !important;
}
.container {
    margin-top: 50px;
}
.pagination {
    margin-top: 20px;
    justify-content: center;
}
.disabled {
    pointer-events: none;
    cursor: default;
}
body {
    background-color: #f8f9fa;
    font-family: Arial, sans-serif;
}
.container {
    margin-top: 50px;
    max-width: 800px;
    background-color: #ffffff;
    border-radius: 5px;
    padding: 20px;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
}
h2 {
    text-align: center;
    margin-bottom: 20px;
}
.table th, .table td {
    vertical-align: middle;
}
.form-control {
    border: 1px solid #ced4da;
    border-radius: 4px;
    position:relative;  
    width: 50%;
}
.btn-primary {
    background-color: #007bff;
    border: none;
}
.btn-primary:hover {
    background-color: #0056b3;
}
.alert {
    margin-top: 20px;
}
.pagination {
    justify-content: center;
}
.disabled {
    pointer-events: none;
    cursor: default;
}
label {
    font-weight: bold;
    position: absolute;

}
.container h1{
    color: black;
    font-size: x-large;
    font-weight: bold;
    font-family: Georgia, 'Times New Roman', Times, serif;
}
.container td, th, tr, label{
    color: black;
    font-size: small;
    font-weight: bold;
    font-family: Georgia, 'Times New Roman', Times, serif;  
}
.form-group{
    color: black;
    font-size: small;
    font-weight: bold;
    font-family: Georgia, 'Times New Roman', Times, serif;  
}
</style>
</head>
<body>
    
<div class="container">
    
    <h1 class="mt-4">Manage Discounts for Products</h1>

    <form method="POST" enctype="multipart/form-data">
        <div class="form-group">
            <label for="company_id">Company</label>
            <select name="company_id" id="company_id" class="form-control" required>
                <?php foreach ($companies as $company): ?>
                    <option value="<?= htmlspecialchars($company['id']) ?>" <?= (isset($discount_to_edit) && $company['id'] == $discount_to_edit['company_id']) ? 'selected' : '' ?>>
                        <?= htmlspecialchars($company['name']) ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="form-group">
            <label>Discount Type</label><br>
            <label><input type="radio" name="discount_type" value="percentage" <?= (isset($discount_to_edit) && $discount_to_edit['discount_type'] == 'percentage') ? 'checked' : '' ?> required> Percentage</label><br>
            <label><input type="radio" name="discount_type" value="fixed_amount" <?= (isset($discount_to_edit) && $discount_to_edit['discount_type'] == 'fixed_amount') ? 'checked' : '' ?> required> Fixed Amount</label>
        <br/>
        </div>

        <div class="form-group">
            <label for="discount_value">Discount Value</label>
            <input type="number" name="discount_value" id="discount_value" class="form-control" step="0.01" value="<?= isset($discount_to_edit) ? htmlspecialchars($discount_to_edit['discount_value']) : '' ?>" required>
        </div>

        <div class="form-group">
            <label for="quantity">Minimum Quantity for Discount</label>
            <input type="number" name="quantity" id="quantity" class="form-control" value="<?= isset($discount_to_edit) ? htmlspecialchars($discount_to_edit['quantity']) : '' ?>" required>
        </div>

        <h2>Select Products</h2>
        <table class="table">
            <thead>
                <tr>
                    <th>Select</th>
                    <th>Image</th>
                    <th>Name</th>
                    <th>Category ID</th>
                    <th>Price</th>
                    <th>Quantity</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($products as $product): ?>
                    <tr>
                        <td>
                            <input type="checkbox" name="product_ids[]" value="<?= htmlspecialchars($product['id']) ?>" <?= isset($product_ids) && in_array($product['id'], $product_ids) ? 'checked' : '' ?>>
                        </td>
                        <td>
                            <?php if (!empty($product['image'])): ?>
                                <img src="<?= htmlspecialchars($product['image']) ?>" alt="Product Image" style="width: 100px; height: auto;">
                            <?php else: ?>
                                No Image
                            <?php endif; ?>
                        </td>
                        <td><?= htmlspecialchars($product['name']) ?></td>
                        <td><?= htmlspecialchars($product['category_id']) ?></td>
                        <td><?= htmlspecialchars($product['price']) ?></td>
                        <td><?= htmlspecialchars($product['quantity']) ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <?php if (isset($discount_to_edit)): ?>
            <input type="hidden" name="discount_id" value="<?= htmlspecialchars($discount_to_edit['id']) ?>">
            <button type="submit" name="update_discount" class="btn btn-primary">Update Discount</button>
        <?php else: ?>
            <button type="submit" name="apply_discount" class="btn btn-primary">Apply Discount to Selected Products</button>
        <?php endif; ?>
    </form>

    <h2 class="mt-4">Existing Discounts</h2>
    <table class="table">
        <thead>
            <tr>
                <th>Company</th>
                <th>Product</th>
                <th>Discount Type</th>
                <th>Discount Value</th>
                <th>Minimum Quantity</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($existing_discounts as $discount): ?>
                <tr>
                    <td><?= htmlspecialchars($discount['company_name']) ?></td>
                    <td><?= htmlspecialchars($discount['product_name']) ?></td>
                    <td><?= htmlspecialchars($discount['discount_type']) ?></td>
                    <td><?= htmlspecialchars($discount['discount_value']) ?></td>
                    <td><?= htmlspecialchars($discount['quantity']) ?></td>
                    <td>
                        <a href="manage_companies.php?edit=<?= htmlspecialchars($discount['id']) ?>" class="btn btn-warning btn-sm">Edit</a>
                        <a href="manage_companies.php?delete=<?= htmlspecialchars($discount['id']) ?>" class="btn btn-danger btn-sm">Delete</a>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>

    <!-- Pagination Links -->
    <nav>
        <ul class="pagination">
            <?php if ($page > 1): ?>
                <li class="page-item">
                    <a class="page-link" href="manage_companies.php?page=<?= $page - 1 ?>" aria-label="Previous">
                        <span aria-hidden="true">&laquo;</span>
                    </a>
                </li>
            <?php endif; ?>

            <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                <li class="page-item <?= $i == $page ? 'active' : '' ?>">
                    <a class="page-link" href="manage_companies.php?page=<?= $i ?>"><?= $i ?></a>
                </li>
            <?php endfor; ?>

            <?php if ($page < $totalPages): ?>
                <li class="page-item">
                    <a class="page-link" href="manage_companies.php?page=<?= $page + 1 ?>" aria-label="Next">
                        <span aria-hidden="true">&raquo;</span>
                    </a>
                </li>
            <?php endif; ?>
        </ul>
    </nav>

    <a href="view_discounts.php" class="btn btn-info">View All Discounts</a>
</div>
</div>
<div class="container"></div>
</body>
</html>
