<?php
session_start();
ob_start(); // Start output buffering
include 'db.php';
include 'header.php';

// Check if the user is logged in and is an Admin
if (!isset($_SESSION['user_id']) || $_SESSION['role_id'] != 1) {
    header('Location: login.php');
    exit;
}

// Handle single order mark as delivered action
if (isset($_POST['mark_delivered'])) {
    $orderId = intval($_POST['mark_delivered']);
    $deliveryDate = date('Y-m-d H:i:s');
    $updateSql = "UPDATE orders SET status = 'Delivered', delivered_on_time = :delivered_on_time WHERE id = :order_id";
    $stmt = $pdo->prepare($updateSql);
    $stmt->bindParam(':delivered_on_time', $deliveryDate);
    $stmt->bindParam(':order_id', $orderId);
    $stmt->execute();
    header('Location: order_processing.php');
    exit;
}

// Handle multiple mark as delivered action
if (isset($_POST['mark_delivered_selected'])) {
    $orderIds = $_POST['order_ids'] ?? [];
    if (!empty($orderIds)) {
        $deliveryDate = date('Y-m-d H:i:s');
        $updateSql = "UPDATE orders SET status = 'Delivered', delivered_on_time = :delivered_on_time WHERE id IN (" . implode(',', array_map('intval', $orderIds)) . ")";
        $stmt = $pdo->prepare($updateSql);
        $stmt->bindParam(':delivered_on_time', $deliveryDate);
        $stmt->execute();
        header('Location: order_processing.php');
        exit;
    }
}

// Handle multiple delete action
if (isset($_POST['delete_selected'])) {
    $orderIds = $_POST['order_ids'] ?? [];
    if (!empty($orderIds)) {
        $deleteSql = "DELETE FROM orders WHERE id IN (" . implode(',', array_map('intval', $orderIds)) . ")";
        $stmt = $pdo->prepare($deleteSql);
        $stmt->execute();
        header('Location: order_processing.php');
        exit;
    }
}

// Pagination settings
$items_per_page = 10; 
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1; 
$offset = ($page - 1) * $items_per_page; 

// Search functionality
$searchTerm = isset($_GET['search']) ? '%' . $_GET['search'] . '%' : '';

// Fetch all pending orders
$sqlPending = "SELECT o.id AS order_id, o.user_id, o.product_id, o.quantity, o.status,
                      p.name AS product_name, p.price AS product_price, p.stock,
                      d.discount_percentage, d.discount_type, d.discount_value,
                      (CASE 
                          WHEN d.discount_type = 'percentage' THEN (p.price * o.quantity * (d.discount_percentage / 100))
                          WHEN d.discount_type = 'fixed_amount' THEN (d.discount_value * o.quantity)
                          ELSE 0
                      END) AS discount_amount,
                      (p.price * o.quantity) - 
                      (CASE 
                          WHEN d.discount_type = 'percentage' THEN (p.price * o.quantity * (d.discount_percentage / 100))
                          WHEN d.discount_type = 'fixed_amount' THEN (d.discount_value * o.quantity)
                          ELSE 0
                      END) AS total_amount,
                      c.name AS company_name,
                      o.created_at
               FROM orders o
               JOIN products p ON o.product_id = p.id
               LEFT JOIN discounts d ON p.id = d.product_id 
               AND d.company_id = (SELECT company_id FROM users WHERE id = o.user_id)
               LEFT JOIN users u ON o.user_id = u.id
               LEFT JOIN companies c ON u.company_id = c.id
               WHERE o.status = 'Pending' AND (p.name LIKE :search OR c.name LIKE :search)
               ORDER BY o.created_at DESC
               LIMIT :limit OFFSET :offset";

$stmtPending = $pdo->prepare($sqlPending);
$stmtPending->bindValue(':search', $searchTerm, PDO::PARAM_STR);
$stmtPending->bindValue(':limit', $items_per_page, PDO::PARAM_INT);
$stmtPending->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmtPending->execute();
$pendingOrders = $stmtPending->fetchAll(PDO::FETCH_ASSOC);

// Total orders for pagination
$totalPendingOrders = $pdo->prepare("SELECT COUNT(*) FROM orders o
                                     JOIN products p ON o.product_id = p.id
                                     LEFT JOIN companies c ON (SELECT company_id FROM users WHERE id = o.user_id) = c.id
                                     WHERE o.status = 'Pending' AND (p.name LIKE :search OR c.name LIKE :search)");
$totalPendingOrders->bindValue(':search', $searchTerm, PDO::PARAM_STR);
$totalPendingOrders->execute();
$totalPendingCount = $totalPendingOrders->fetchColumn();
$totalPendingPages = ceil($totalPendingCount / $items_per_page);

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Admin Dashboard</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
    <link rel="stylesheet" href="./style/css/clogin.css">
<style>
    
body {
    background-color: #0e5091 !important;
}
.container {
    margin-top: 50px;
}
.pagination {
    margin-top: 20px;
    justify-content: center;
}
.disabled {
    pointer-events: none;
    cursor: default;
}
body {
    background-color: #f8f9fa;
    font-family: Arial, sans-serif;
}
.container {
    margin-top: 50px;
    max-width: 800px;
    background-color: #ffffff;
    border-radius: 5px;
    padding: 20px;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
}
h2 {
    text-align: center;
    margin-bottom: 20px;
}
.table th, .table td {
    vertical-align: middle;
}
.form-control {
    border: 1px solid #ced4da;
    border-radius: 4px;
    position:relative;  
    width: 50%;
}
.btn-primary {
    background-color: #007bff;
    border: none;
}
.btn-primary:hover {
    background-color: #0056b3;
}
.alert {
    margin-top: 20px;
}
.pagination {
    justify-content: center;
}
.disabled {
    pointer-events: none;
    cursor: default;
}
label {
    font-weight: bold;
    position: absolute;

}
.container h1{
    color: black;
    font-size: x-large;
    font-weight: bold;
    font-family: Georgia, 'Times New Roman', Times, serif;
}
.container td, th, tr, label{
    color: black;
    font-size: small;
    font-weight: bold;
    font-family: Georgia, 'Times New Roman', Times, serif;  
}
.form-group{
    color: black;
    font-size: small;
    font-weight: bold;
    font-family: Georgia, 'Times New Roman', Times, serif;  
}
</style>

</head>
<body>
<div class="container">
    <div class="row">
        <div class="col-md-12">
            <h1 class="mt-4">View Orders</h1>

            <form method="GET" class="mb-3">
                <input type="text" name="search" placeholder="Search by product or company" value="<?= htmlspecialchars($_GET['search'] ?? '') ?>" class="form-control" style="width: 300px; display: inline-block;">
                <button type="submit" class="btn btn-primary">Search</button>
            </form>
            <a href="Order_delivered.php" class="btn btn-primary">Order Delivered</a>
            <form method="POST">
                <h3>Pending Orders</h3>
                <table class="table">
                    <thead>
                        <tr>
                            <th><input type="checkbox" id="select_all"></th> <!-- Select All Checkbox -->
                            <th>Product Name</th>
                            <th>Company Name</th>
                            <th>Quantity</th>
                            <th>Status</th>
                            <th>Total Amount</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($pendingOrders)): ?>
                            <tr>
                                <td colspan="7">No pending orders found.</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($pendingOrders as $order): ?>
                                <tr>
                                    <td><input type="checkbox" name="order_ids[]" value="<?= htmlspecialchars($order['order_id']) ?>"></td>
                                    <td><?= htmlspecialchars($order['product_name']) ?></td>
                                    <td><?= htmlspecialchars($order['company_name']) ?></td>
                                    <td><?= htmlspecialchars($order['quantity']) ?></td>
                                    <td><?= htmlspecialchars($order['status']) ?></td>
                                    <td><?= htmlspecialchars(number_format($order['total_amount'], 2)) ?> Rs.</td>
                                    <td>
                                        <button type="submit" name="mark_delivered" value="<?= htmlspecialchars($order['order_id']) ?>" class="btn btn-success">Delivered</button>
                                        <button type="submit" name="delete_order" value="<?= htmlspecialchars($order['order_id']) ?>" class="btn btn-danger">Delete</button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>

                <!-- Bulk Actions -->
                <button type="submit" name="mark_delivered_selected" class="btn btn-success">Process All</button>
                <button type="submit" name="delete_selected" class="btn btn-danger">Delete All</button>
            </form>

            <!-- Pagination for Pending Orders -->
            <nav>
                <ul class="pagination">
                    <?php for ($i = 1; $i <= $totalPendingPages; $i++): ?>
                        <li class="page-item <?= ($i == $page) ? 'active' : '' ?>">
                            <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($_GET['search'] ?? '') ?>"><?= $i ?></a>
                        </li>
                    <?php endfor; ?>
                </ul>
            </nav>

        </div>
    </div>
</div>

<!-- JavaScript to Select All Checkboxes -->
<script>
    document.getElementById('select_all').addEventListener('click', function() {
        let checkboxes = document.querySelectorAll('input[name="order_ids[]"]');
        for (let checkbox of checkboxes) {
            checkbox.checked = this.checked;
        }
    });
</script>
<div class="container"></div>
</body>
</html>
