<?php
session_start();
include 'db.php';
include 'header.php';

// Check if the user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// Get search parameters with default values
$company_id = isset($_GET['company']) ? $_GET['company'] : null;
$from_date = isset($_GET['from_date']) ? $_GET['from_date'] : null;
$to_date = isset($_GET['to_date']) ? $_GET['to_date'] : null;
$search_query = isset($_GET['search']) ? $_GET['search'] : '';

// Ensure we don’t pass null values to urlencode
$urlEncodedToDate = urlencode($to_date ?? '');
$urlEncodedFromDate = urlencode($from_date ?? '');
$urlEncodedSearchQuery = urlencode($search_query ?? '');

// Build SQL query based on search criteria
$sqlOrders = "SELECT o.*, u.company_id, p.name AS product_name 
              FROM orders o 
              JOIN users u ON o.user_id = u.id 
              JOIN products p ON o.product_id = p.id 
              WHERE 1=1";

$params = [];

if ($company_id) {
    $sqlOrders .= " AND u.company_id = :company_id";
    $params['company_id'] = $company_id;
}

if ($from_date && $to_date) {
    $sqlOrders .= " AND DATE(o.created_at) BETWEEN :from_date AND :to_date";
    $params['from_date'] = $from_date;
    $params['to_date'] = $to_date;
} elseif ($from_date) {
    $sqlOrders .= " AND DATE(o.created_at) >= :from_date";
    $params['from_date'] = $from_date;
} elseif ($to_date) {
    $sqlOrders .= " AND DATE(o.created_at) <= :to_date";
}

// Pagination settings
$limit = 10; // Number of orders per page
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$page = max(1, $page); // Ensure page is at least 1
$start = ($page - 1) * $limit;

$sqlOrders .= " LIMIT $start, $limit";

$stmtOrders = $pdo->prepare($sqlOrders);
$stmtOrders->execute($params);
$orders = $stmtOrders->fetchAll(PDO::FETCH_ASSOC);

// Get total number of records
$sqlCount = "SELECT COUNT(*) FROM orders o JOIN users u ON o.user_id = u.id WHERE 1=1";
if ($company_id) $sqlCount .= " AND u.company_id = :company_id";
if ($from_date && $to_date) $sqlCount .= " AND DATE(o.created_at) BETWEEN :from_date AND :to_date";
elseif ($from_date) $sqlCount .= " AND DATE(o.created_at) >= :from_date";
elseif ($to_date) $sqlCount .= " AND DATE(o.created_at) <= :to_date";

$stmtCount = $pdo->prepare($sqlCount);
$stmtCount->execute($params);
$totalRecords = $stmtCount->fetchColumn();
$totalPages = ceil($totalRecords / $limit);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Orders List</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <style>
    
    body {
        background-color: #0e5091 !important;
    }
    .container {
        margin-top: 50px;
    }
    .pagination {
        margin-top: 20px;
        justify-content: center;
    }
    .disabled {
        pointer-events: none;
        cursor: default;
    }
    body {
        background-color: #f8f9fa;
        font-family: Arial, sans-serif;
    }
    .container {
        margin-top: 50px;
        max-width: 800px;
        background-color: #ffffff;
        border-radius: 5px;
        padding: 20px;
        box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
    }
    h2 {
        text-align: center;
        margin-bottom: 20px;
    }
    .table th, .table td {
        vertical-align: middle;
    }
    .form-control {
        border: 1px solid #ced4da;
        border-radius: 4px;
        position:relative;  
        width: 50%;
    }
    .btn-primary {
        background-color: #007bff;
        border: none;
    }
    .btn-primary:hover {
        background-color: #0056b3;
    }
    .alert {
        margin-top: 20px;
    }
    .pagination {
        justify-content: center;
    }
    .disabled {
        pointer-events: none;
        cursor: default;
    }
    label {
        font-weight: bold;
        position: absolute;
    
    }
    .container h1{
        color: black;
        font-size: x-large;
        font-weight: bold;
        font-family: Georgia, 'Times New Roman', Times, serif;
    }
    .container td, th, tr, label{
        color: black;
        font-size: small;
        font-weight: bold;
        font-family: Georgia, 'Times New Roman', Times, serif;  
        position: relative;
    }
    .form-group{
        color: black;
        font-size: small;
        font-weight: bold;
        font-family: Georgia, 'Times New Roman', Times, serif;  
    }
    lable{
        color: black;
        font-size: small;
        font-weight: bold;
        font-family: Georgia, 'Times New Roman', Times, serif;  
        position: relative;
    }
    </style>
</head>
<body>
<div class="container mt-4">
    <h1>Orders List</h1>
    
    <form method="post" action="generate_invoice.php">
        <div class="form-group">
            <label for="order_ids">Select Orders for Invoice:</label>
            <select name="order_ids[]" id="order_ids" class="form-control" multiple>
                <?php foreach ($orders as $order): ?>
                    <option value="<?= htmlspecialchars($order['id'] ?? '', ENT_QUOTES, 'UTF-8') ?>">
                        Order ID: <?= htmlspecialchars($order['id'] ?? '', ENT_QUOTES, 'UTF-8') ?> - 
                        Product: <?= htmlspecialchars($order['product_name'] ?? '', ENT_QUOTES, 'UTF-8') ?> - 
                        Total: Rs <?= htmlspecialchars(number_format($order['total_amount'] ?? 0, 2)) ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        
        <input type="hidden" name="company" value="<?= htmlspecialchars($company_id ?? '', ENT_QUOTES, 'UTF-8') ?>">
        <input type="hidden" name="from_date" value="<?= htmlspecialchars($from_date ?? '', ENT_QUOTES, 'UTF-8') ?>">
        <input type="hidden" name="to_date" value="<?= htmlspecialchars($to_date ?? '', ENT_QUOTES, 'UTF-8') ?>">
        
        <button type="submit" name="generate_invoice" class="btn btn-success">Generate Invoice</button>
    </form>
    
    <table class="table mt-3">
        <thead>
            <tr>
                <th>Order ID</th>
                <th>Product Name</th>
                <th>Quantity</th>
                <th>Total Amount (Rs)</th>
                <th>Date</th>
                <th>Company ID</th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($orders)): ?>
                <tr>
                    <td colspan="6">No orders found.</td>
                </tr>
            <?php else: ?>
                <?php foreach ($orders as $order): ?>
                    <tr>
                        <td><?= htmlspecialchars($order['id'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
                        <td><?= htmlspecialchars($order['product_name'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
                        <td><?= htmlspecialchars($order['quantity'] ?? 0, ENT_QUOTES, 'UTF-8') ?></td>
                        <td><?= htmlspecialchars(number_format($order['total_amount'] ?? 0, 2)) ?></td>
                        <td><?= htmlspecialchars($order['created_at'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
                        <td><?= htmlspecialchars($order['company_id'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>

    <nav>
        <ul class="pagination">
            <?php if ($page > 1): ?>
                <li class="page-item">
                    <a class="page-link" href="search_orders.php?page=<?= $page - 1 ?>&company=<?= urlencode($company_id ?? '') ?>&from_date=<?= urlencode($from_date ?? '') ?>&to_date=<?= urlencode($to_date ?? '') ?>">Previous</a>
                </li>
            <?php endif; ?>
            <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                <li class="page-item<?= $i === $page ? ' active' : '' ?>">
                    <a class="page-link" href="search_orders.php?page=<?= $i ?>&company=<?= urlencode($company_id ?? '') ?>&from_date=<?= urlencode($from_date ?? '') ?>&to_date=<?= urlencode($to_date ?? '') ?>"><?= $i ?></a>
                </li>
            <?php endfor; ?>
            <?php if ($page < $totalPages): ?>
                <li class="page-item">
                    <a class="page-link" href="search_orders.php?page=<?= $page + 1 ?>&company=<?= urlencode($company_id ?? '') ?>&from_date=<?= urlencode($from_date ?? '') ?>&to_date=<?= urlencode($to_date ?? '') ?>">Next</a>
                </li>
            <?php endif; ?>
        </ul>
    </nav>
</div>
<div class="container"></div>
</body>
</html>
