<?php
session_start();

// Secure session configurations
ini_set('session.cookie_secure', 1);
ini_set('session.cookie_httponly', 1);
ini_set('session.use_strict_mode', 1);
session_regenerate_id(true);

include __DIR__ . '/db.php'; // Use absolute paths
require_once __DIR__ . '/vendor/autoload.php'; // Required for TCPDF

// Check if the user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// Validate and sanitize user_id
$user_id = filter_var($_SESSION['user_id'], FILTER_VALIDATE_INT);

if (!$user_id) {
    error_log('Invalid user ID: ' . $_SESSION['user_id']);
    header('Location: error_page.php');
    exit;
}

// Fetch user details securely
$sqlUser = "SELECT * FROM users WHERE id = ?";
$stmtUser = $pdo->prepare($sqlUser);
$stmtUser->execute([$user_id]);
$user = $stmtUser->fetch(PDO::FETCH_ASSOC);

if (!$user) {
    error_log('User not found: ' . $user_id);
    header('Location: error_page.php');
    exit;
}

// Assuming company_id is part of the users table
$company_id = $user['company_id'];

// Fetch today's orders for the company
$sqlOrders = "SELECT o.*, u.company_id FROM orders o 
              JOIN users u ON o.user_id = u.id 
              WHERE u.company_id = :company_id 
              AND DATE(o.created_at) = CURDATE()";

$stmtOrders = $pdo->prepare($sqlOrders);
$stmtOrders->execute(['company_id' => $company_id]);

$orders = $stmtOrders->fetchAll(PDO::FETCH_ASSOC);

// Check if orders exist for the company on the current date
if (empty($orders)) {
    echo "No orders found for today.";
    exit;
}

// Generate the invoice using TCPDF
$pdf = new TCPDF();
$pdf->AddPage();

// Add company logo, name, and date
$pdf->Image(__DIR__ . '/path_to_logo.png', 10, 10, 20);
$pdf->SetFont('helvetica', 'B', 16);
$pdf->Cell(0, 15, 'Your Company Name', 0, 1, 'C');
$pdf->SetFont('helvetica', '', 12);
$pdf->Cell(0, 15, 'Date: ' . date('Y-m-d H:i:s'), 0, 1, 'R');

// Add order details
$pdf->SetFont('helvetica', '', 10);
$pdf->Ln(10);
$pdf->Cell(40, 10, 'Order ID', 1);
$pdf->Cell(50, 10, 'Product Name', 1);
$pdf->Cell(30, 10, 'Quantity', 1);
$pdf->Cell(30, 10, 'Total Amount (Rs)', 1);
$pdf->Ln();

$totalAmount = 0;

foreach ($orders as $order) {
    $pdf->Cell(40, 10, htmlspecialchars($order['id'], ENT_QUOTES, 'UTF-8'), 1);
    $pdf->Cell(50, 10, htmlspecialchars($order['product_name'], ENT_QUOTES, 'UTF-8'), 1);
    $pdf->Cell(30, 10, htmlspecialchars($order['quantity'], ENT_QUOTES, 'UTF-8'), 1);
    $pdf->Cell(30, 10, htmlspecialchars($order['total_amount'], ENT_QUOTES, 'UTF-8'), 1);
    $pdf->Ln();
    
    $totalAmount += $order['total_amount'];
}

// Add total amount
$pdf->Ln(10);
$pdf->SetFont('helvetica', 'B', 12);
$pdf->Cell(0, 15, 'Total Amount: Rs. ' . number_format($totalAmount, 2), 0, 1, 'R');

// Output the PDF
$pdf->Output('invoice.pdf', 'I');
?>
