<?php
session_start();
include 'db.php';
include 'header.php';

// Check if the user is logged in and is an Administrator
if (!isset($_SESSION['user_id']) || $_SESSION['role_id'] != 1) {
    header('Location: login.php');
    exit;
}

// Handle search functionality
$search_query = isset($_GET['search']) ? $_GET['search'] : '';

// Pagination setup
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$perPage = 10; // Number of items per page
$offset = ($page - 1) * $perPage;

// Fetch existing discounts with pagination and search functionality
$sql = "SELECT d.*, c.name AS company_name, p.name AS product_name 
        FROM discounts d
        JOIN companies c ON d.company_id = c.id
        JOIN products p ON d.product_id = p.id
        WHERE c.name LIKE :search 
           OR p.name LIKE :search 
           OR d.discount_percentage LIKE :search
        LIMIT :offset, :perPage";
$stmt = $pdo->prepare($sql);
$stmt->bindValue(':search', "%$search_query%", PDO::PARAM_STR);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->bindValue(':perPage', $perPage, PDO::PARAM_INT);
$stmt->execute();
$discounts = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch total count of discounts for pagination
$sqlCount = "SELECT COUNT(*) 
              FROM discounts d
              JOIN companies c ON d.company_id = c.id
              JOIN products p ON d.product_id = p.id
              WHERE c.name LIKE :search 
                 OR p.name LIKE :search 
                 OR d.discount_percentage LIKE :search";
$stmtCount = $pdo->prepare($sqlCount);
$stmtCount->bindValue(':search', "%$search_query%", PDO::PARAM_STR);
$stmtCount->execute();
$total = $stmtCount->fetchColumn();
$totalPages = ceil($total / $perPage);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>View Discounts</title>
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
</head>
<body>
<div class="container">
    <h1 class="mt-4">View Discounts</h1>

    <!-- Search Form -->
    <form method="GET" class="mb-4">
        <div class="form-group">
            <label for="search">Search Discounts</label>
            <input type="text" name="search" id="search" class="form-control" value="<?= htmlspecialchars($search_query) ?>">
        </div>
        <button type="submit" class="btn btn-primary">Search</button>
    </form>

    <!-- Discounts Table -->
    <table class="table">
        <thead>
            <tr>
                <th>Company</th>
                <th>Product</th>
                <th>Discount Percentage</th>
                <th>Minimum Quantity</th>
            </tr>
        </thead>
        <tbody>
            <?php if (count($discounts) > 0): ?>
                <?php foreach ($discounts as $discount): ?>
                    <tr>
                        <td><?= htmlspecialchars($discount['company_name']) ?></td>
                        <td><?= htmlspecialchars($discount['product_name']) ?></td>
                        <td><?= htmlspecialchars($discount['discount_percentage']) ?>%</td>
                        <td><?= htmlspecialchars($discount['quantity']) ?></td>
                    </tr>
                <?php endforeach; ?>
            <?php else: ?>
                <tr>
                    <td colspan="4">No discounts found.</td>
                </tr>
            <?php endif; ?>
        </tbody>
    </table>

    <!-- Pagination Links -->
    <nav>
        <ul class="pagination">
            <?php if ($page > 1): ?>
                <li class="page-item">
                    <a class="page-link" href="?page=<?= $page - 1 ?>&search=<?= htmlspecialchars($search_query) ?>" aria-label="Previous">
                        <span aria-hidden="true">&laquo;</span>
                    </a>
                </li>
            <?php endif; ?>

            <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                <li class="page-item <?= $i == $page ? 'active' : '' ?>">
                    <a class="page-link" href="?page=<?= $i ?>&search=<?= htmlspecialchars($search_query) ?>"><?= $i ?></a>
                </li>
            <?php endfor; ?>

            <?php if ($page < $totalPages): ?>
                <li class="page-item">
                    <a class="page-link" href="?page=<?= $page + 1 ?>&search=<?= htmlspecialchars($search_query) ?>" aria-label="Next">
                        <span aria-hidden="true">&raquo;</span>
                    </a>
                </li>
            <?php endif; ?>
        </ul>
    </nav>
</div>
<div class="container"></div>
</body>
</html>
